﻿/*
VERSION:	2.9
2.9		Fixed VOW.runSequence() to always return a promise.
2.8		Added VOW.preApply()
2.7		Added promise.getValue()
2.6		Added VOW.wrap() which accepts an async function that resolves using a callback, and allows that function to be used within a promise-chain
2.5		Added VOW.runSequence() which calls an array of asynchronous functions in sequence, and returns a promise for the completion of the sequence.
2.4		Fixed VOW.firstWait() to not default to 1 second upon invalid input.
2.3		Fixed VOW.wait() so that received promise-chain values pass through it to the next function in the chain.

2.2		Fixed VOW.wait() so that it actually works within promise-chains.

2.1		Added getValue() to promise objects so that their resolved values can be read.

2.0		Added VOW.start() so a promise-chain can be started using any functions, like so:
				VOW.start()
				.then( normalFunc1 )
				.then( asyncFunc2 )
				.then( lastFunc, failFunc )

1.9		Fixed VOW.every() to handle empty arrays.  Now if there's nothing to wait for, then resume successfully instead of failing.

1.8		Fixed a memory leak:  where using my_prom.then() on completed promises used to re-call all of its previous listeners

1.7		Added vow.getStatus() so external code can determine the state of an un-sent promise

1.5		Revised forEach() to be a strictly local function instead of polluting the global Array object
			(It was conflicting with ZigoEngine)

1.4		Add:  VOW.firstWait() & VOW.wait()
	VOW.firstWait(500)
		.then( func )
		.then( VOW.wait(1000) )
		.then( func )
	
	func()
		.then( VOW.wait(1000) )
		.then( func )
		
1.3		VOW.wait has a default wait of 1 frame AKA 34 milliseconds
1.2		VOW.wait() added to allow setTimeout-style delays, like so:
				VOW.wait( 1000 ).then( doSomething );
1.1		keep() and doBreak() return their promise obj, to allow returning pre-kept promises like so:
				return VOW.make().keep()
				This allows async functions to be optionally be synchronous
	
	
FUNCTIONS
	VOW				Used to create vows & check promises
		make
		every
		any
		first
		kept
		broken
		wait
		firstWait
		start
		runSequence
		wrap
		
	vow				Used by async functions that make promises
		promise
		keep
		doBreak
		getStatus
		getValue
	
	promise		Used to call reactions later when some async thing finishes
		is_promise
		then
		getStatus
		
				
USAGE:
	#include "functions/VOW.as"
	_global.VOW = VOW;
	function doAsync( input ){
		var vow = VOW.make();
		onSucceed = function(){
			vow.keep( results );
		}
		onFail = function(){
			vow.doBreak( reason );
		}
		return vow.promise;
	}// doAsync()
	
	doAsync("param A").then( useResults )
	
	
	var promiseState = vow.getStatus();		// "pending"  "kept"  "broken"
	
	
	promiseA = doAsync("param A");
	promiseB = doSomethingElse("param B");
	promiseC = doAsync("param C");
	VOW.every([
	 promiseA,
	 promiseB,
	 promiseC
	]).then( laterFunc );
	
	
	
EXPLANATION:
	function doAsync()
		returns a promise and calls promise.keep() later on
	
	promise1.then( onSucceess, onFail );
		When this promise is kept, it calls the 1st function
		If this promise is broken, it calls the 2nd function
	
	VOW.every([ promise1, promise2 ]).then( handleResults );
	
	promise1
		.then( handleResult1, onFail )	// use the result of promise1, then make promise2
		.then( handleResult2, onFail )	// use the result of promise2, then make promise3
		.then( handleResult3, onFail )	// use the result of promise3
		

CRED:
	This promise library is a derivative work.
	It was adapted from Douglas Crockford's JavaScript implementation, located here:
	https://github.com/douglascrockford/monad/blob/master/vow.js
	... and then tweaked and extended.
	Kudos to him for introducing me to this brilliant concept!
*/



var vow_scope = function(){
	
	function forEach( array, callback){
		for(var i=0; i<array.length; i++){
			callback( array[i], i );
		}// for:  each item within this array
	}// forEach()
	
	
	
	function enlighten(queue, fate){
		forEach(queue, function (func) {
			setTimeout(function(){
				func(fate);
			}, 0);// setTimeout()
		});// forEach()
	}// enlighten()
	
	
	return {
		make: function (){
			var breakers = [],
					fate = undefined,
					keepers = [], 
					status = 'pending';
			
			
			function enqueue (resolution, func, vow){
				var queue = resolution === 'kept' ? keepers : breakers;
				queue[queue.length] = typeof func !== 'function'
					? vow[resolution]
					: function (value) {
						try {
							var result = func(value);
							if(result && result.is_promise === true){
								result.then( vow.keep, vow.doBreak );
							}// if:  result is a promise
							else
							{// if:  result is NOT a promise
								vow.keep(result);
							}// if:  result is NOT a promise
						}// try
						catch(e){
							vow.doBreak(e);
						}// catch
					}// queue[length] = func
			}// enqueue()
			
			
			function herald (state, value, queue){
				if(status !== 'pending'){
					 throw 'overpromise';		// do not resolve twice
				}// if:  this promise is already resolved
				fate = value;
				status = state;
				enlighten(queue, fate);
				//keepers.length = 0;
				//breakers.length = 0;
				keepers = [];
				breakers = [];
			}// herald()
			
			
			return {
				getStatus: function(){
					return status;
				},// getStatus()
				getValue: function(){
					return fate;
				},// getValue()
				doBreak: function (value){
					herald('broken', value, breakers);
					return this.promise;
				},// doBreak()
				keep: function (value){
					herald('kept', value, keepers);
					return this.promise;
				},// keep()
				promise: {
					is_promise: true,
					then: function (kept, broken){
						var vow = VOW.make();
						switch (status){
							case 'pending':
								enqueue('kept', kept, vow);
								enqueue('break', broken, vow);
							break;
							case 'kept':
								enqueue('kept', kept, vow);
								enlighten(keepers, fate);
								keepers = [];
							break;
							case 'broken':
								enqueue('break', broken, vow);
								enlighten(breakers, fate);
								breakers = [];
							break;
						}// case: status
						return vow.promise;
					},// then()
					getStatus: function(){
						return status;
					},// getStatus()
					getValue: function(){
						return fate;
					}// getValue()
				}// {promise}
			}// return {break, keep, promise}
		},// make()
		
		
// The every function takes an array of promises and returns a promise that
// will deliver an array of results only if every promise is kept.
// 
// The "every" function returns a promise whose then() gives an array containing
// the result of each promise ONLY if all promises succeeded.
		every: function (array){
			var remaining = array.length,
					results = [],
					vow = VOW.make();
			if(!remaining){
				vow.keep([]);
			}// if:  array is empty
			else
			{// if:  array has values
				forEach( array, function (promise, i){
					promise.then(
						function (value) {
							results[i] = value;
							remaining--;
							if(remaining === 0){
								vow.keep(results);
							}// if:  array counter is Zero
						},// success()
						function (reason) {
							remaining = null;
							vow.doBreak(reason);
						}// fail()
					);// then()
				});// array.forEach()
				
			}// if:  array has values
			return vow.promise;
		},// every()
		
		
// The first function takes an array of promises and returns a promise to
// deliver the first observed kept promise, or a broken promise if all of
// the promises are broken.
// 
// The "first" function returns a promise whose then() gives the result 
// of the promise that completes first, ignoring all others.
		first: function (array){
			var found = false,
					remaining = array.length,
					vow = VOW.make();
			function check(){
				remaining--;
				if(remaining === 0  &&  !found){
					vow.doBreak();
				}// if:  no more items to check
			}// check()
			if(remaining === 0){
				vow.doBreak(array);
			}// if:  array was empty to begin with
			else
			{// if:  array has values
				forEach( array, function (promise){
					promise.then(
						function (value){
							if(!found){
								found = true;
								vow.keep(value);
							}//if:  found not updated yet
							check();
						},// success()
						check		// fail()
					);// then()
				});// array.forEach()
			}// if:  array has values
			return vow.promise;
		},// first()
		
		
// The any function takes an array of promises and returns a promise that
// will deliver a possibly sparse array of results of any kept promises.
// The result will contain an undefined element for each broken promise.
// 
// The "any" function returns a promise whose then() gives an array containing
// the results of each promise. Promises that failed have an undefined value as
// their result.
		any: function (array){
			var remaining = array.length,
					results = [],
					vow = VOW.make();
			function check(){
				remaining--;
				if(remaining === 0){
					vow.keep(results);
				}// if:  no more items to check
			}// check()
			if(!remaining){
				vow.keep(results);
			}// if:  array was empty to begin with
			else
			{// if:  array has values
				forEach( array, function (promise, i) {
					promise.then(
						function (value) {
							results[i] = value;
							check();
						},// success()
						check		// fail()
					);// then()
				});// array.forEach()
			}// if:  array has values
			return vow.promise;
		},// any()
		
		
// The VOW.kept() function immediately returns a successfully kept promise.
// This can be useful for testing purposes.
// This is also useful for beginning a promise chain, like so:
// 
// VOW.kept()
// .then( step1 )
// .then( step2 )
// .then( step3 )
		kept: function (value){
			var vow = VOW.make();
			vow.keep(value);
			return vow.promise;
		},// kept()
		
		
// The VOW.broken() function immediately returns a broken promise.
// This can be useful for testing purposes.
		broken: function (reason){
			var vow = VOW.make();
			vow.doBreak(reason);
			return vow.promise;
		},// broken()
		
		
// The VOW.wait() function is used to cause a pause in a promise chain.
// wait() accepts a number representing how long to wait.
// This function can be used like so:
// 
// step1()
// .then( step2 )
// .then( VOW.wait(1000) )
// .then( step3 )
		wait: function ( milliseconds ){
			var doWait = function( input ){
				var vow = VOW.make();
				var milliseconds = (milliseconds===undefined) ? 0 : milliseconds;
				setTimeout(function() {
					vow.keep(input);
					},
					milliseconds);
				return vow.promise;
			}// doWait()
			return doWait;
		},// wait()
		
		
// The VOW.firstWait() function is used to cause a delay, before starting a promise chain.
// firstWait() accepts a number representing how long to wait.
// This function can be used like so:
// 
// VOW.firstWait( 1000 )
// .then( step1 )
// .then( step2 )
// .then( step3 )
		firstWait: function (milliseconds){
			var vow = VOW.make();
			var milliseconds = (milliseconds===undefined) ? 0 : milliseconds;
			setTimeout(function() {
				vow.keep({});
				},
				milliseconds);
			return vow.promise;
		},// firstWait()
		
		
// The VOW.start() function immediately returns a successfully kept promise.
// This function is identical to VOW.kept()...
// ... and exists for the purpose of clearly expressing intent when starting a promise chain, like so:
// 
// VOW.start()
// .then( step1 )
// .then( step2 )
// .then( step3 )
		start: function (value){
			var vow = VOW.make();
			vow.keep(value);
			return vow.promise;
		},// start()
		
		
// the VOW.runSequence() function takes an array of async functions (which return promises) and returns a promise that
// represents the successful completion of the entire sequence.
// 
// When runSequence() is called, it immediately begins running the first function in the sequence.
		runSequence: function ( sequence ){
			var prevPromise = VOW.make().keep();
			// var prevPromise = null;
			for(var s=0; s<sequence.length; s++){
				var thisStep = sequence[s];
				var newProm = prevPromise.then( thisStep );
				if( newProm.is_promise !== true )		newProm = VOW.kept();
				prevPromise = newProm;
			}// for:  each step in the sequence
			var finalPromise = prevPromise;
			return finalPromise;
		},// runSequence()
		
		
// The VOW.wrap() function takes a function reference, and an index number
// and outputs a function that gives a promise when called.
// The promise will deliver the first parameter passed to the callback function.
		wrap: function ( func, callbackIndex ){
			if( callbackIndex === undefined )		var callbackIndex = 0;
			return function(){
				var vow = VOW.make();
				
				if( callbackIndex === undefined )		var callbackIndex = arguments.length-1;
				
				arguments[callbackIndex] = function( output ){
					vow.keep( output );
				}
				func.apply( this, arguments );
				return vow.promise;
			}// return func ()
		},// wrap()
		
		
// The VOW.preApply() function takes a function reference, and a single parameter and outputs a function.
// The parameter is stored at the time of definition.
// When this new function is run, it'll call the specified function and pass the stored parameter.
		preApply: function( func, param ){
			return function(){
				return func( param );
			}// return func()
		},// preApply()
		
		null:null
	}// return {}
};// vow_scope()
var VOW = vow_scope();
delete vow_scope;